import axios from 'axios';

const API_BASE_URL = '/plataforma/api';

// Configuração do axios
const api = axios.create({
  baseURL: API_BASE_URL,
  headers: {
    'Content-Type': 'application/json',
  },
});

// Interceptor para adicionar token se existir
api.interceptors.request.use((config) => {
  const token = localStorage.getItem('rideshare_token');
  if (token) {
    config.headers.Authorization = `Bearer ${token}`;
  }
  return config;
});

// Interceptor para tratar erros
api.interceptors.response.use(
  (response) => response,
  (error) => {
    if (error.response?.status === 401) {
      localStorage.removeItem('rideshare_token');
      localStorage.removeItem('rideshare_user');
      window.location.reload();
    }
    return Promise.reject(error);
  }
);

// Serviços de Autenticação
export const authService = {
  loginUser: (email, senha) =>
    api.post('/auth.php', { action: 'login_usuario', email, senha }),
  
  loginDriver: (email, senha) =>
    api.post('/auth.php', { action: 'login_motorista', email, senha }),
  
  registerUser: (userData) =>
    api.post('/auth.php', { action: 'register_usuario', ...userData }),
  
  registerDriver: (driverData) =>
    api.post('/auth.php', { action: 'register_motorista', ...driverData }),
  
  verifyUser: (user_id) =>
    api.post('/auth.php', { action: 'verificar_usuario', user_id }),
};

// Serviços de Corridas
export const rideService = {
  requestRide: (rideData) =>
    api.post('/corridas.php', { action: 'solicitar_corrida', ...rideData }),
  
  cancelRide: (corrida_id, usuario_id, motivo = 'Cancelado pelo usuário') =>
    api.post('/corridas.php', {
      action: 'cancelar_corrida',
      corrida_id,
      cancelado_por: 'usuario',
      usuario_id,
      motivo,
    }),
  
  getRide: (corrida_id) =>
    api.get(`/corridas.php?action=obter_corrida&corrida_id=${corrida_id}`),
  
  listRides: (tipo_usuario, usuario_id, status = '') =>
    api.get(`/corridas.php?action=listar_corridas&tipo_usuario=${tipo_usuario}&usuario_id=${usuario_id}&status=${status}`),
  
  rateRide: (corrida_id, avaliacao, comentario = '', avaliado_por = 'usuario') =>
    api.post('/corridas.php', {
      action: 'avaliar_corrida',
      corrida_id,
      avaliacao,
      comentario,
      avaliado_por,
    }),
};

// Serviços de Motorista
export const driverService = {
  updateLocation: (motorista_id, latitude, longitude) =>
    api.post('/motoristas.php', {
      action: 'atualizar_localizacao',
      motorista_id,
      latitude,
      longitude,
    }),
  
  toggleAvailability: (motorista_id, disponivel) =>
    api.post('/motoristas.php', {
      action: 'alterar_disponibilidade',
      motorista_id,
      disponivel,
    }),
  
  getBalance: (motorista_id) =>
    api.get(`/motoristas.php?action=obter_saldo&motorista_id=${motorista_id}`),
  
  getDayStats: (motorista_id) =>
    api.get(`/motoristas.php?action=obter_estatisticas_dia&motorista_id=${motorista_id}`),
  
  getAvailableRides: (motorista_id) =>
    api.get(`/corridas.php?action=buscar_corridas_disponiveis&motorista_id=${motorista_id}`),
  
  acceptRide: (corrida_id, motorista_id) =>
    api.post('/corridas.php', {
      action: 'aceitar_corrida',
      corrida_id,
      motorista_id,
    }),
  
  startRide: (corrida_id, motorista_id) =>
    api.post('/corridas.php', {
      action: 'iniciar_corrida',
      corrida_id,
      motorista_id,
    }),
  
  completeRide: (corrida_id, motorista_id) =>
    api.post('/corridas.php', {
      action: 'concluir_corrida',
      corrida_id,
      motorista_id,
    }),
};

// Serviços de Admin
export const adminService = {
  getCategories: () =>
    api.get('/admin.php?action=listar_categorias'),
  
  getAvailableDrivers: (categoria_veiculo_id, latitude, longitude, raio_km = 10) =>
    api.get(`/corridas.php?action=motoristas_disponiveis&categoria_veiculo_id=${categoria_veiculo_id}&latitude=${latitude}&longitude=${longitude}&raio_km=${raio_km}`),
};

export default api;