import React, { useState, useEffect } from 'react';
import { Car, Clock, DollarSign, Users, Package, MapPin, X } from 'lucide-react';

const VehicleSelectionModal = ({
  isOpen,
  onClose,
  categories,
  origin,
  destination,
  originCoords,
  destinationCoords,
  onRideRequest,
  loading
}) => {
  const [selectedCategory, setSelectedCategory] = useState(null);
  const [routeInfo, setRouteInfo] = useState(null);
  const [calculatingRoute, setCalculatingRoute] = useState(false);

  useEffect(() => {
    if (isOpen && originCoords && destinationCoords) {
      calculateRoute();
    }
  }, [isOpen, originCoords, destinationCoords]);

  const calculateRoute = () => {
    setCalculatingRoute(true);
    
    const service = new window.google.maps.DistanceMatrixService();
    service.getDistanceMatrix({
      origins: [originCoords],
      destinations: [destinationCoords],
      travelMode: window.google.maps.TravelMode.DRIVING,
      unitSystem: window.google.maps.UnitSystem.METRIC,
      avoidHighways: false,
      avoidTolls: false
    }, (response, status) => {
      setCalculatingRoute(false);
      
      if (status === 'OK') {
        const element = response.rows[0].elements[0];
        if (element.status === 'OK') {
          const distance = element.distance.value / 1000; // em km
          const duration = Math.round(element.duration.value / 60); // em minutos
          
          setRouteInfo({ distance, duration });
        }
      }
    });
  };

  const calculatePrice = (category) => {
    if (!routeInfo) return 0;
    
    const basePrice = parseFloat(category.tarifa_base);
    const pricePerKm = parseFloat(category.tarifa_por_km);
    const pricePerMinute = parseFloat(category.tarifa_por_minuto);
    
    let totalPrice = basePrice + (routeInfo.distance * pricePerKm) + (routeInfo.duration * pricePerMinute);
    
    // Aplicar tarifa dinâmica (simulada)
    const surge = getSurgeMultiplier();
    if (surge > 1) {
      totalPrice *= surge;
    }
    
    return totalPrice;
  };

  const getSurgeMultiplier = () => {
    const hour = new Date().getHours();
    if ((hour >= 7 && hour <= 9) || (hour >= 17 && hour <= 19)) {
      return 1.5; // Horário de pico
    }
    return 1.0;
  };

  const getVehicleIcon = (type) => {
    switch (type) {
      case 'moto':
        return '🏍️';
      case 'carro':
        return '🚗';
      case 'van':
        return '🚐';
      default:
        return '🚗';
    }
  };

  const handleCategorySelect = (category) => {
    setSelectedCategory(category);
  };

  const handleConfirmRide = () => {
    if (selectedCategory) {
      const price = calculatePrice(selectedCategory);
      onRideRequest(selectedCategory.id, price);
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-50 bg-black bg-opacity-50 flex items-end">
      <div className="w-full bg-white rounded-t-3xl max-h-[80vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-100">
          <h2 className="text-xl font-bold text-gray-900">Escolha seu veículo</h2>
          <button
            onClick={onClose}
            className="w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center hover:bg-gray-200 transition-colors"
          >
            <X size={20} className="text-gray-600" />
          </button>
        </div>

        <div className="p-6 space-y-6 overflow-y-auto max-h-[calc(80vh-80px)]">
          {/* Informações da Rota */}
          {!calculatingRoute && routeInfo && (
            <div className="bg-gray-50 rounded-2xl p-4">
              <div className="flex items-center justify-between">
                <div className="text-center">
                  <p className="text-sm text-gray-600">Distância</p>
                  <p className="text-lg font-bold text-gray-900">{routeInfo.distance.toFixed(1)} km</p>
                </div>
                <div className="w-px h-8 bg-gray-300"></div>
                <div className="text-center">
                  <p className="text-sm text-gray-600">Tempo</p>
                  <p className="text-lg font-bold text-gray-900">{routeInfo.duration} min</p>
                </div>
              </div>
            </div>
          )}

          {/* Lista de Veículos */}
          {calculatingRoute ? (
            <div className="flex flex-col items-center justify-center py-12">
              <div className="w-8 h-8 border-2 border-black border-t-transparent rounded-full animate-spin mb-4"></div>
              <p className="text-gray-600">Calculando preços...</p>
            </div>
          ) : (
            <div className="space-y-3">
              {categories.map((category) => {
                const price = calculatePrice(category);
                const isSelected = selectedCategory?.id === category.id;
                const surge = getSurgeMultiplier();
                
                return (
                  <button
                    key={category.id}
                    onClick={() => handleCategorySelect(category)}
                    className={`w-full p-4 rounded-2xl border-2 transition-all duration-200 ${
                      isSelected
                        ? 'border-black bg-black text-white'
                        : 'border-gray-200 bg-white hover:border-gray-300'
                    }`}
                  >
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-4">
                        <div className="text-3xl">{getVehicleIcon(category.tipo)}</div>
                        <div className="text-left">
                          <div className="flex items-center gap-2">
                            <h3 className={`font-semibold ${isSelected ? 'text-white' : 'text-gray-900'}`}>
                              {category.nome}
                            </h3>
                            {surge > 1 && (
                              <span className="bg-orange-500 text-white text-xs px-2 py-1 rounded-full">
                                {surge}x
                              </span>
                            )}
                          </div>
                          <p className={`text-sm ${isSelected ? 'text-gray-300' : 'text-gray-600'}`}>
                            {category.descricao}
                          </p>
                          {category.capacidade_passageiros && (
                            <p className={`text-xs ${isSelected ? 'text-gray-400' : 'text-gray-500'}`}>
                              Até {category.capacidade_passageiros} passageiros
                            </p>
                          )}
                        </div>
                      </div>
                      
                      <div className="text-right">
                        <p className={`text-lg font-bold ${isSelected ? 'text-white' : 'text-gray-900'}`}>
                          R$ {price.toFixed(2)}
                        </p>
                        {surge > 1 && (
                          <p className={`text-xs ${isSelected ? 'text-gray-400' : 'text-gray-500'}`}>
                            Tarifa dinâmica
                          </p>
                        )}
                      </div>
                    </div>
                  </button>
                );
              })}
            </div>
          )}

          {/* Detalhes da Viagem */}
          <div className="bg-gray-50 rounded-2xl p-4 space-y-3">
            <div className="flex items-start gap-3">
              <div className="w-3 h-3 bg-blue-500 rounded-full mt-2"></div>
              <div>
                <p className="text-xs text-gray-600 font-medium">ORIGEM</p>
                <p className="text-sm text-gray-900">{origin}</p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="w-3 h-3 bg-red-500 rounded-full mt-2"></div>
              <div>
                <p className="text-xs text-gray-600 font-medium">DESTINO</p>
                <p className="text-sm text-gray-900">{destination}</p>
              </div>
            </div>
          </div>

          {/* Botão de Confirmar */}
          <button
            onClick={handleConfirmRide}
            disabled={!selectedCategory || loading || calculatingRoute}
            className="w-full bg-black text-white py-4 rounded-2xl font-semibold text-lg disabled:opacity-50 disabled:cursor-not-allowed hover:bg-gray-800 transition-colors"
          >
            {loading ? (
              <div className="flex items-center justify-center gap-2">
                <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                Solicitando...
              </div>
            ) : selectedCategory ? (
              `Solicitar ${selectedCategory.nome}`
            ) : (
              'Selecione um veículo'
            )}
          </button>
        </div>
      </div>
    </div>
  );
};

export default VehicleSelectionModal;