import React, { useEffect, useRef } from 'react';

const GoogleMap = ({ 
  center, 
  originCoords, 
  destinationCoords, 
  onAddressSelect 
}) => {
  const mapRef = useRef(null);
  const mapInstanceRef = useRef(null);
  const directionsRendererRef = useRef(null);

  useEffect(() => {
    if (mapRef.current && window.google) {
      initializeMap();
    }
  }, []);

  useEffect(() => {
    if (mapInstanceRef.current && center) {
      mapInstanceRef.current.setCenter({
        lat: center.latitude,
        lng: center.longitude
      });
    }
  }, [center]);

  useEffect(() => {
    if (originCoords && destinationCoords) {
      drawRoute();
    }
  }, [originCoords, destinationCoords]);

  const initializeMap = () => {
    const mapOptions = {
      center: {
        lat: center?.latitude || -23.550520,
        lng: center?.longitude || -46.633308
      },
      zoom: 13,
      styles: [
        {
          featureType: 'poi',
          elementType: 'labels',
          stylers: [{ visibility: 'off' }]
        }
      ],
      disableDefaultUI: true,
      zoomControl: true,
      mapTypeControl: false,
      streetViewControl: false,
      fullscreenControl: false,
    };

    mapInstanceRef.current = new window.google.maps.Map(mapRef.current, mapOptions);

    // Adicionar marcador da localização atual
    if (center) {
      new window.google.maps.Marker({
        position: {
          lat: center.latitude,
          lng: center.longitude
        },
        map: mapInstanceRef.current,
        icon: {
          url: 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png',
        },
        title: 'Sua localização'
      });
    }
  };

  const drawRoute = () => {
    if (!mapInstanceRef.current || !originCoords || !destinationCoords) return;

    // Limpar rota anterior
    if (directionsRendererRef.current) {
      directionsRendererRef.current.setMap(null);
    }

    const directionsService = new window.google.maps.DirectionsService();
    directionsRendererRef.current = new window.google.maps.DirectionsRenderer({
      suppressMarkers: false,
      polylineOptions: {
        strokeColor: '#3b82f6',
        strokeWeight: 4,
        strokeOpacity: 0.8
      }
    });

    directionsRendererRef.current.setMap(mapInstanceRef.current);

    directionsService.route({
      origin: {
        lat: originCoords.latitude,
        lng: originCoords.longitude
      },
      destination: {
        lat: destinationCoords.latitude,
        lng: destinationCoords.longitude
      },
      travelMode: window.google.maps.TravelMode.DRIVING
    }, (result, status) => {
      if (status === 'OK') {
        directionsRendererRef.current.setDirections(result);
      } else {
        console.error('Erro ao calcular rota:', status);
      }
    });
  };

  return (
    <div 
      ref={mapRef} 
      className="w-full h-full"
      style={{ minHeight: '400px' }}
    />
  );
};

export default GoogleMap;