import { useState, useEffect, createContext, useContext } from 'react';
import { authService } from '../services/api';

const AuthContext = createContext();

export const useAuth = () => {
  const context = useContext(AuthContext);
  if (!context) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
};

export const AuthProvider = ({ children, userType = 'user' }) => {
  const [user, setUser] = useState(null);
  const [loading, setLoading] = useState(true);
  const [isAuthenticated, setIsAuthenticated] = useState(false);

  useEffect(() => {
    checkAuthStatus();
  }, []);

  const checkAuthStatus = async () => {
    try {
      const savedUser = localStorage.getItem('rideshare_user');
      const loginTime = localStorage.getItem('rideshare_login_time');
      
      if (savedUser && loginTime) {
        const timeDiff = Date.now() - parseInt(loginTime);
        const oneDay = 24 * 60 * 60 * 1000;
        
        if (timeDiff < oneDay) {
          const userData = JSON.parse(savedUser);
          
          // Verificar se usuário ainda é válido
          const response = await authService.verifyUser(userData.id);
          
          if (response.data.success && response.data.usuario_ativo) {
            setUser(userData);
            setIsAuthenticated(true);
          } else {
            logout();
          }
        } else {
          logout();
        }
      }
    } catch (error) {
      console.error('Erro ao verificar autenticação:', error);
      logout();
    } finally {
      setLoading(false);
    }
  };

  const login = async (email, senha) => {
    try {
      const response = userType === 'driver' 
        ? await authService.loginDriver(email, senha)
        : await authService.loginUser(email, senha);
      
      if (response.data.success) {
        const userData = response.data[userType === 'driver' ? 'motorista' : 'usuario'];
        
        setUser(userData);
        setIsAuthenticated(true);
        
        localStorage.setItem('rideshare_user', JSON.stringify(userData));
        localStorage.setItem('rideshare_login_time', Date.now().toString());
        
        if (response.data.token) {
          localStorage.setItem('rideshare_token', response.data.token);
        }
        
        return { success: true, user: userData };
      } else {
        return { success: false, message: response.data.message };
      }
    } catch (error) {
      console.error('Erro no login:', error);
      return { 
        success: false, 
        message: error.response?.data?.message || 'Erro de conexão' 
      };
    }
  };

  const register = async (userData) => {
    try {
      const response = userType === 'driver'
        ? await authService.registerDriver(userData)
        : await authService.registerUser(userData);
      
      if (response.data.success) {
        return { success: true, message: response.data.message };
      } else {
        return { success: false, message: response.data.message };
      }
    } catch (error) {
      console.error('Erro no registro:', error);
      return { 
        success: false, 
        message: error.response?.data?.message || 'Erro de conexão' 
      };
    }
  };

  const logout = () => {
    setUser(null);
    setIsAuthenticated(false);
    localStorage.removeItem('rideshare_user');
    localStorage.removeItem('rideshare_login_time');
    localStorage.removeItem('rideshare_token');
    localStorage.removeItem('rideshare_corrida_ativa');
  };

  const value = {
    user,
    loading,
    isAuthenticated,
    login,
    register,
    logout,
  };

  return (
    <AuthContext.Provider value={value}>
      {children}
    </AuthContext.Provider>
  );
};