import React, { useState, useEffect } from 'react';
import { 
  Power, 
  User, 
  LogOut,
  Menu,
  Moon,
  Sun,
  Search,
  CheckCircle,
  XCircle,
  Settings,
  Smartphone
} from 'lucide-react';
import GoogleMap from './GoogleMap';
import EarningsScreen from './EarningsScreen';
import { useGeolocation } from '../../hooks/useGeolocation';
import { useSound } from '../../hooks/useSound';
import { driverService } from '../../services/api';

const MainScreen = ({ user, onLogout }) => {
  const { location, error: locationError } = useGeolocation();
  const { playNewRideSound, playCancelSound, playSuccessSound, stopCurrentSound } = useSound();
  
  const [isOnline, setIsOnline] = useState(false);
  const [balance, setBalance] = useState(0.00);
  const [loading, setLoading] = useState(false);
  const [notification, setNotification] = useState(null);
  const [isDarkMap, setIsDarkMap] = useState(false);
  const [showSideMenu, setShowSideMenu] = useState(false);
  const [currentRide, setCurrentRide] = useState(null);
  const [rideTimer, setRideTimer] = useState(30);
  const [connectingAnimation, setConnectingAnimation] = useState(false);
  const [showOverlayPermission, setShowOverlayPermission] = useState(false);
  const [dailyEarnings, setDailyEarnings] = useState(0.00);
  const [dailyRides, setDailyRides] = useState(0);
  const [currentScreen, setCurrentScreen] = useState('main'); // 'main', 'earnings'


  useEffect(() => {
    loadInitialData();
  }, []);

  // Effect para atualizar localização periodicamente quando online
  useEffect(() => {
    let locationInterval;
    
    if (isOnline && user?.id && location) {
      console.log('📍 Iniciando atualização de localização');
      
      // Atualizar localização a cada 30 segundos
      locationInterval = setInterval(async () => {
        try {
          await driverService.updateLocation(user.id, location.latitude, location.longitude);
          console.log('📍 Localização atualizada');
        } catch (error) {
          console.warn('Erro ao atualizar localização:', error);
        }
      }, 30000);
      
      return () => {
        clearInterval(locationInterval);
      };
    }
  }, [isOnline, user?.id, location]);

  // Effect para gerenciar corridas reais quando online
  useEffect(() => {
    let rideInterval;
    let statsInterval;
    
    if (isOnline && user?.id) {
      console.log('🟢 Motorista online - buscando corridas reais');
      
      // Primeira busca em 3 segundos
      const firstRideTimeout = setTimeout(() => {
        if (isOnline && !currentRide) {
          console.log('🚗 Primeira busca por corridas...');
          fetchAvailableRides();
        }
      }, 3000);
      
      // Buscar corridas a cada 10 segundos
      rideInterval = setInterval(() => {
        if (isOnline && !currentRide) {
          console.log('🔄 Buscando corridas disponíveis...');
          fetchAvailableRides();
        }
      }, 10000);
      
      // Atualizar estatísticas a cada 30 segundos quando online
      statsInterval = setInterval(() => {
        if (isOnline) {
          console.log('📊 Atualizando estatísticas...');
          loadInitialData();
        }
      }, 30000);
      
      return () => {
        clearTimeout(firstRideTimeout);
        clearInterval(rideInterval);
        clearInterval(statsInterval);
      };
    } else {
      console.log('🔴 Motorista offline - parando busca de corridas');
      if (rideInterval) {
        clearInterval(rideInterval);
      }
      if (statsInterval) {
        clearInterval(statsInterval);
      }
    }
  }, [isOnline, currentRide, user?.id]);

  const loadInitialData = async () => {
    try {
      if (user?.id) {
        // Carregar saldo real do motorista
        const balanceResponse = await driverService.getBalance(user.id);
        if (balanceResponse.data.success) {
          setBalance(parseFloat(balanceResponse.data.saldo) || 0.00);
        }
        
        // Carregar estatísticas do dia (não precisamos mais exibir no header)
        const statsResponse = await driverService.getDayStats(user.id);
        if (statsResponse.data.success) {
          setDailyEarnings(parseFloat(statsResponse.data.ganho_hoje) || 0.00);
          setDailyRides(parseInt(statsResponse.data.corridas_hoje) || 0);
        }
      } else {
        // Fallback para dados simulados
        setBalance(150.75);
      }
    } catch (error) {
      console.error('Erro ao carregar dados:', error);
      setBalance(0.00);
    }
  };

  // Atualizar status do motorista no servidor
  const updateDriverStatus = async (isOnline) => {
    if (!user?.id || !location) return;
    
    try {
      // Atualizar localização
      await driverService.updateLocation(user.id, location.latitude, location.longitude);
      
      // Atualizar disponibilidade
      await driverService.toggleAvailability(user.id, isOnline);
      
      console.log(`📍 Status atualizado: ${isOnline ? 'Online' : 'Offline'}`);
    } catch (error) {
      console.error('Erro ao atualizar status do motorista:', error);
      throw error;
    }
  };

  const toggleOnlineStatus = async () => {
    console.log('🔘 BOTÃO CONECTAR CLICADO - Status atual:', isOnline);
    console.log('📍 Localização disponível:', location);
    console.log('👤 Usuário:', user?.id);
    
    // Temporariamente removendo verificação de localização para debug
    // if (!location) {
    //   console.warn('❌ Localização não disponível');
    //   showNotification('Aguarde a localização', 'Obtendo sua localização...', 'warning');
    //   return;
    // }

    const newStatus = !isOnline;
    console.log('🔄 Novo status será:', newStatus ? 'ONLINE' : 'OFFLINE');
    
    if (newStatus) {
      // Animação de conectando
      setConnectingAnimation(true);
      setLoading(true);
      
      // Simular processo de conexão com animação
      setTimeout(async () => {
        try {
          // Atualizar status no servidor
          if (user?.id) {
            await updateDriverStatus(newStatus);
          }
          
          setIsOnline(newStatus);
          setConnectingAnimation(false);
          
          playSuccessSound();
          showNotification(
            'Conectado!', 
            'Você está online e buscando corridas reais',
            'success'
          );
          
          console.log('🟢 MOTORISTA CONECTADO - Buscando corridas reais');
        } catch (error) {
          console.error('Erro ao alterar status:', error);
          showNotification('Erro ao conectar', 'Tente novamente', 'error');
          setConnectingAnimation(false);
        } finally {
          setLoading(false);
        }
      }, 2000); // 2 segundos de animação
    } else {
      // Desconectar imediatamente
      setLoading(true);
      
      try {
        if (user?.id) {
          await updateDriverStatus(newStatus);
        }
        
        setIsOnline(newStatus);
        setCurrentRide(null);
        
        try {
          playCancelSound();
        } catch (error) {
          console.warn('Erro ao tocar som:', error);
        }
        
        showNotification(
          'Desconectado', 
          'Você está offline',
          'info'
        );
        
        console.log('🔴 MOTORISTA DESCONECTADO');
      } catch (error) {
        console.error('Erro ao desconectar:', error);
        showNotification('Erro ao desconectar', 'Tente novamente', 'error');
      } finally {
        setLoading(false);
      }
    }
  };

  // Buscar corridas reais disponíveis
  const fetchAvailableRides = async () => {
    if (!user?.id) {
      console.warn('❌ ID do motorista não disponível');
      return;
    }
    
    try {
      console.log('🔍 Buscando corridas disponíveis para motorista:', user.id);
      
      const response = await driverService.getAvailableRides(user.id);
      
      if (response.data.success && response.data.corridas.length > 0) {
        const ride = response.data.corridas[0]; // Pegar a primeira corrida disponível
        
        console.log('✅ CORRIDA REAL ENCONTRADA:', ride);
        
        // Converter para formato esperado pelo componente
        const formattedRide = {
          id: ride.id,
          origem_endereco: ride.origem_endereco,
          destino_endereco: ride.destino_endereco,
          valor_total: parseFloat(ride.valor_total).toFixed(2),
          distancia_km: parseFloat(ride.distancia_km).toFixed(1),
          duracao_minutos: ride.duracao_minutos,
          usuario_nome: ride.usuario_nome,
          usuario_telefone: ride.usuario_telefone,
          isReal: true // Flag para identificar corrida real
        };
        
        setCurrentRide(formattedRide);
        setRideTimer(10); // 10 segundos para aceitar corrida
        
        // Tocar som de nova corrida
        try {
          playNewRideSound();
          console.log('🔊 Som de nova corrida tocado');
        } catch (error) {
          console.warn('❌ Erro ao tocar som:', error);
        }
        
        // Iniciar countdown
        startRideCountdown(true);
        
        showNotification(
          'Nova Corrida Real!', 
          `${ride.usuario_nome} precisa de você`,
          'success'
        );
      } else {
        console.log('ℹ️ Nenhuma corrida disponível no momento');
      }
    } catch (error) {
      console.error('❌ Erro ao buscar corridas:', error);
      showNotification('Erro', 'Falha ao buscar corridas', 'error');
    }
  };

  const startRideCountdown = (isRealRide = false) => {
    console.log('⏰ Iniciando countdown da corrida');
    
    let timeLeft = 10; // 10 segundos para todas as corridas
    const countdownInterval = setInterval(() => {
      timeLeft--;
      setRideTimer(timeLeft);
      console.log(`⏱️ Tempo restante: ${timeLeft}s`);
      
      if (timeLeft <= 0) {
        clearInterval(countdownInterval);
        console.log('⏰ CORRIDA EXPIROU!');
        
        // Parar som da corrida
        stopCurrentSound();
        
        setCurrentRide(null);
        
        try {
          playCancelSound();
        } catch (error) {
          console.warn('Erro ao tocar som de expiração:', error);
        }
        
        showNotification('Corrida expirou', 'Aguardando nova solicitação', 'warning');
      }
    }, 1000);
  };



  const handleAcceptRide = async (rideId) => {
    console.log('✅ ACEITANDO CORRIDA:', rideId);
    
    // Parar som imediatamente
    stopCurrentSound();
    
    if (!user?.id) {
      showNotification('Erro', 'ID do motorista não disponível', 'error');
      return;
    }
    
    try {
      // Se for corrida real, aceitar via API
      if (currentRide?.isReal) {
        setLoading(true);
        
        const response = await driverService.acceptRide(rideId, user.id);
        
        if (response.data.success) {
          console.log('✅ CORRIDA REAL ACEITA COM SUCESSO');
          
          // Atualizar status do motorista para indisponível
          await driverService.toggleAvailability(user.id, false);
          
          setCurrentRide(null);
          setIsOnline(false); // Ficar offline após aceitar corrida
          
          try {
            playSuccessSound();
          } catch (error) {
            console.warn('Erro ao tocar som de sucesso:', error);
          }
          
          showNotification(
            'Corrida aceita!', 
            'Dirija-se ao local de origem. Você ficou offline.',
            'success'
          );
          
          // Recarregar dados após aceitar corrida (para atualizar estatísticas se necessário)
          setTimeout(() => {
            loadInitialData();
          }, 1000);
        } else {
          throw new Error(response.data.message || 'Erro ao aceitar corrida');
        }
      } else {
        // Corrida simulada (comportamento antigo)
        setCurrentRide(null);
        
        try {
          playSuccessSound();
        } catch (error) {
          console.warn('Erro ao tocar som de sucesso:', error);
        }
        
        showNotification('Corrida aceita!', 'Dirija-se ao local de origem', 'success');
      }
    } catch (error) {
      console.error('❌ Erro ao aceitar corrida:', error);
      showNotification('Erro', 'Falha ao aceitar corrida: ' + error.message, 'error');
    } finally {
      setLoading(false);
    }
  };

  const handleRejectRide = (rideId) => {
    console.log('❌ CORRIDA REJEITADA:', rideId);
    
    // Parar som imediatamente
    stopCurrentSound();
    
    setCurrentRide(null);
    
    try {
      playCancelSound();
    } catch (error) {
      console.warn('Erro ao tocar som de cancelamento:', error);
    }
    
    const message = currentRide?.isReal ? 
      'Corrida real recusada. Buscando outras...' : 
      'Corrida recusada. Aguardando nova solicitação';
    
    showNotification('Corrida recusada', message, 'info');
  };

  const showNotification = (title, message, type = 'info') => {
    setNotification({ title, message, type });
    setTimeout(() => setNotification(null), 5000);
  };

  // Função para solicitar permissão de sobreposição
  const requestOverlayPermission = () => {
    if ('Notification' in window) {
      Notification.requestPermission().then(permission => {
        if (permission === 'granted') {
          showNotification('Permissão concedida!', 'Notificações ativadas', 'success');
        } else {
          showNotification('Permissão negada', 'Ative nas configurações do navegador', 'warning');
        }
      });
    }
    
    // Para dispositivos móveis - tentar ativar modo fullscreen
    if (document.documentElement.requestFullscreen) {
      document.documentElement.requestFullscreen().catch(() => {
        showNotification('Modo tela cheia', 'Não foi possível ativar', 'info');
      });
    }
    
    setShowOverlayPermission(false);
  };

  // Renderizar tela de ganhos se selecionada
  if (currentScreen === 'earnings') {
    return (
      <EarningsScreen 
        user={user} 
        onBack={() => setCurrentScreen('main')} 
      />
    );
  }

  return (
    <div className="relative h-screen bg-gray-100">
      {/* Google Maps */}
      <GoogleMap
        center={location || { latitude: -23.550520, longitude: -46.633308 }}
        driverLocation={location}
        availableRides={[]}
        isOnline={isOnline}
        isDarkMap={isDarkMap}
        showSurgeHexagons={isOnline}
      />

      {/* Header */}
      <div className="absolute top-0 left-0 right-0 z-20 bg-white shadow-md">
        <div className="flex items-center justify-between p-4">
          <div className="flex items-center gap-4">
            <button
              onClick={() => setShowSideMenu(true)}
              className="w-10 h-10 bg-gray-100 rounded-xl flex items-center justify-center hover:bg-gray-200 transition-colors"
            >
              <Menu size={20} className="text-gray-600" />
            </button>
            <div className="flex items-center gap-2">
              <div 
                className="w-8 h-8 rounded-lg flex items-center justify-center"
                style={{ backgroundColor: '#FF6B35' }}
              >
                <User size={16} className="text-white" />
              </div>
              <span 
                className="font-bold text-lg"
                style={{ color: '#FF6B35' }}
              >
                RideShare
              </span>
            </div>
          </div>
          
          <div className="flex items-center gap-3">
            <button
              onClick={() => setIsDarkMap(!isDarkMap)}
              className={`w-10 h-10 rounded-xl flex items-center justify-center transition-all ${
                isDarkMap 
                  ? 'bg-gray-800 text-white' 
                  : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
              }`}
            >
              {isDarkMap ? <Sun size={18} /> : <Moon size={18} />}
            </button>
            

            
            <div className={`flex items-center gap-2 px-4 py-2 rounded-full text-sm font-semibold ${
              isOnline 
                ? 'bg-green-100 text-green-700 border border-green-200' 
                : 'bg-gray-100 text-gray-600 border border-gray-200'
            }`}>
              <div className={`w-2 h-2 rounded-full ${isOnline ? 'bg-green-500' : 'bg-gray-400'}`} />
              {isOnline ? 'Online' : 'Offline'}
            </div>
          </div>
        </div>
      </div>



      {/* Erro de Localização */}
      {locationError && (
        <div className="absolute top-32 left-4 right-4 z-10">
          <div className="bg-red-500 bg-opacity-90 text-white rounded-xl p-3 mx-auto max-w-sm">
            <div className="flex items-center gap-2">
              <span className="text-lg">📍</span>
              <div>
                <div className="font-semibold text-sm">Erro de Localização</div>
                <div className="text-xs opacity-90">{locationError}</div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Status Notification */}
      {isOnline && !locationError && (
        <div className="absolute top-32 left-4 right-4 z-10">
          <div className="bg-black bg-opacity-80 text-white rounded-xl p-4 mx-auto max-w-sm">
            <div className="flex items-start gap-3">
              <div 
                className="w-6 h-6 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5"
                style={{ backgroundColor: '#FF6B35' }}
              >
                <Search size={12} />
              </div>
              <div>
                <div className="font-semibold text-sm">Buscando corridas reais</div>
                <div className="text-xs opacity-90 mt-1">Conectado ao sistema de corridas</div>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Bottom Panel */}
      <div className="absolute bottom-0 left-0 right-0 z-20 bg-black bg-opacity-80 backdrop-blur-sm p-5">
        <div className="text-center">
          {/* Status de Busca quando Online */}
          {isOnline && !connectingAnimation && (
            <div className="text-white text-lg font-semibold flex items-center justify-center gap-3 mb-4">
              <div className="relative">
                <Search size={20} className="animate-spin" />
                <div className="absolute -inset-1 bg-orange-500 rounded-full opacity-20 animate-ping" />
              </div>
              <span className="animate-pulse">Buscando corridas</span>
              <div className="flex gap-1">
                <div className="w-1.5 h-1.5 bg-white rounded-full animate-bounce" />
                <div className="w-1.5 h-1.5 bg-white rounded-full animate-bounce" style={{ animationDelay: '0.1s' }} />
                <div className="w-1.5 h-1.5 bg-white rounded-full animate-bounce" style={{ animationDelay: '0.2s' }} />
              </div>
            </div>
          )}
          
          {/* Botão Principal - Conectar/Desconectar */}
          <div className="relative">
            {connectingAnimation && (
              <>
                <div className="absolute inset-0 rounded-full animate-connecting-wave" style={{ border: '2px solid rgba(255, 107, 53, 0.6)' }} />
                <div className="absolute inset-0 rounded-full animate-connecting-wave" style={{ border: '2px solid rgba(255, 107, 53, 0.4)', animationDelay: '0.5s' }} />
                <div className="absolute inset-0 rounded-full animate-connecting-wave" style={{ border: '2px solid rgba(255, 107, 53, 0.2)', animationDelay: '1s' }} />
              </>
            )}
            
            <button
              onClick={toggleOnlineStatus}
              disabled={loading}
              className={`relative font-bold text-lg px-16 py-4 rounded-full flex items-center justify-center gap-3 mx-auto transition-all duration-300 min-w-48 shadow-lg transform ${
                connectingAnimation ? 'scale-110 animate-pulse-glow' : 'scale-100 hover:scale-105'
              }`}
              style={{ 
                backgroundColor: '#FF6B35',
                color: 'white',
                opacity: loading ? 0.9 : 1,
                cursor: loading ? 'not-allowed' : 'pointer',
                boxShadow: connectingAnimation ? '0 0 30px rgba(255, 107, 53, 0.6)' : '0 4px 15px rgba(255, 107, 53, 0.4)'
              }}
            >
              {connectingAnimation ? (
                <>
                  <div className="relative">
                    <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
                    <div className="absolute inset-0 w-5 h-5 border-2 border-transparent border-b-white rounded-full animate-spin" style={{ animationDirection: 'reverse', animationDuration: '0.8s' }} />
                  </div>
                  <span className="animate-pulse">Conectando...</span>
                </>
              ) : loading ? (
                <>
                  <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
                  <span>Processando...</span>
                </>
              ) : isOnline ? (
                <>
                  <Power size={20} className="transition-transform duration-300" />
                  <span>Desconectar</span>
                </>
              ) : (
                <>
                  <Power size={20} className="transition-transform duration-300 hover:rotate-180" />
                  <span>Conectar</span>
                </>
              )}
            </button>
          </div>
        </div>
      </div>

      {/* Side Menu */}
      {showSideMenu && (
        <div className="fixed inset-0 z-50">
          <div 
            className="absolute inset-0 bg-black bg-opacity-50"
            onClick={() => setShowSideMenu(false)}
          />
          <div className="absolute top-0 left-0 w-72 h-full bg-white">
            <div 
              className="text-white p-6 pt-12"
              style={{ background: 'linear-gradient(135deg, #FF6B35 0%, #FF8A65 100%)' }}
            >
              <div className="flex items-center gap-4">
                <div className="w-15 h-15 bg-white bg-opacity-20 rounded-full flex items-center justify-center">
                  <User size={24} />
                </div>
                <div>
                  <div className="font-semibold text-lg">{user?.nome || 'Motorista'}</div>
                  <div className="text-sm opacity-90">⭐ 4.8</div>
                </div>
              </div>
            </div>
            
            <div className="p-0">
              <div 
                className="flex items-center gap-4 p-5 hover:bg-gray-50 cursor-pointer"
                onClick={() => {
                  setCurrentScreen('earnings');
                  setShowSideMenu(false);
                }}
              >
                <div className="w-6 text-gray-500">💰</div>
                <span>Ganhos</span>
              </div>
              <div className="flex items-center gap-4 p-5 hover:bg-gray-50 cursor-pointer">
                <div className="w-6 text-gray-500">📊</div>
                <span>Histórico</span>
              </div>
              <div className="flex items-center gap-4 p-5 hover:bg-gray-50 cursor-pointer">
                <div className="w-6 text-gray-500">⚙️</div>
                <span>Configurações</span>
              </div>
              <div 
                className="flex items-center gap-4 p-5 hover:bg-gray-50 cursor-pointer"
                onClick={onLogout}
              >
                <LogOut size={16} className="text-gray-500" />
                <span>Sair</span>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Ride Request Panel - ESTILO UBER */}
      {currentRide && (
        <div className="fixed inset-0 z-40 bg-black bg-opacity-70 flex items-center justify-center p-4">
          <div className="bg-gray-800 rounded-2xl shadow-2xl w-full max-w-sm mx-auto transform animate-slide-up text-white">
            {/* Header com avaliação e badges */}
            <div className="p-5 pb-3">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-2">
                  <span className="text-lg font-bold">4.8 ⭐</span>
                  <span className="text-gray-300">Recorrente</span>
                </div>
                {currentRide.isReal && (
                  <span className="bg-green-500 text-white text-xs px-2 py-1 rounded-full font-bold">
                    REAL
                  </span>
                )}
              </div>
              
              {/* Badges de verificação */}
              <div className="flex gap-2 mb-4">
                <span className="bg-blue-500 text-white text-xs px-2 py-1 rounded-full">
                  ✓ Cartão Verif.
                </span>
                <span className="bg-blue-500 text-white text-xs px-2 py-1 rounded-full">
                  ✓ CPF
                </span>
              </div>
              
              {/* Valor principal */}
              <div className="flex items-end justify-between mb-2">
                <div>
                  <span className="text-4xl font-bold">R${parseFloat(currentRide.valor_total).toFixed(2)}</span>
                </div>
                <div className="text-right">
                  <div className="text-yellow-400 text-lg font-bold">R$2,20</div>
                  <div className="text-yellow-400 text-xs">Por Km</div>
                </div>
              </div>
              
              {/* Multiplicadores */}
              <div className="flex items-center gap-4 mb-4">
                <div className="flex items-center gap-1 text-sm">
                  <span className="w-5 h-5 bg-gray-600 rounded-full flex items-center justify-center text-xs">⚡</span>
                  <span>x2.0</span>
                </div>
                <div className="flex items-center gap-1 text-sm">
                  <span className="w-5 h-5 bg-gray-600 rounded-full flex items-center justify-center text-xs">⭐</span>
                  <span>+R$15.5</span>
                </div>
              </div>
              
              {/* Bônus */}
              <div className="flex items-center gap-2 mb-4 text-sm">
                <span className="w-4 h-4 bg-gray-600 rounded-full flex items-center justify-center text-xs">▶</span>
                <span>Corrida Longa R$5.0 bônus</span>
              </div>
            </div>
            
            {/* Endereços */}
            <div className="px-5 pb-5">
              {/* Origem */}
              <div className="flex items-start gap-3 mb-3">
                <div className="w-3 h-3 bg-green-500 rounded-full mt-2 flex-shrink-0"></div>
                <div className="flex-1">
                  <div className="text-sm font-medium mb-1">
                    {currentRide.duracao_minutos}Min • {currentRide.distancia_km}Km
                  </div>
                  <div className="text-gray-300 text-sm leading-tight">
                    {currentRide.origem_endereco}
                  </div>
                </div>
              </div>
              
              {/* Destino */}
              <div className="flex items-start gap-3 mb-6">
                <div className="w-3 h-3 bg-orange-500 rounded-full mt-2 flex-shrink-0"></div>
                <div className="flex-1">
                  <div className="text-sm font-medium mb-1">
                    32Min • 6,3km
                  </div>
                  <div className="text-gray-300 text-sm leading-tight">
                    {currentRide.destino_endereco}
                  </div>
                </div>
              </div>
              
              {/* Botão Aceitar com timer */}
              <div className="relative">
                <button
                  onClick={() => handleAcceptRide(currentRide.id)}
                  className="w-full bg-yellow-400 hover:bg-yellow-500 text-black font-bold text-lg py-4 rounded-full transition-all duration-200 transform hover:scale-105 shadow-lg flex items-center justify-center"
                >
                  Aceitar
                </button>
                
                {/* Timer circular */}
                <div className="absolute right-4 top-1/2 transform -translate-y-1/2 w-10 h-10 border-2 border-black rounded-full flex items-center justify-center bg-yellow-400">
                  <span className="text-black font-bold text-sm">{rideTimer}s</span>
                </div>
              </div>
              
              {/* Botão recusar pequeno */}
              <button
                onClick={() => handleRejectRide(currentRide.id)}
                className="w-full mt-3 py-2 text-gray-400 text-sm hover:text-white transition-colors"
              >
                Recusar
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Botão Flutuante - Permissões */}
      <button
        onClick={() => setShowOverlayPermission(true)}
        className="fixed bottom-24 right-4 z-30 w-14 h-14 text-white rounded-full shadow-2xl flex items-center justify-center transition-all duration-300 transform hover:scale-110"
        style={{ 
          backgroundColor: '#FF6B35',
          boxShadow: '0 8px 25px rgba(255, 107, 53, 0.4)',
          animation: 'float 3s ease-in-out infinite'
        }}
        title="Configurar Permissões"
      >
        <Settings size={24} className="animate-pulse" />
      </button>

      {/* Modal de Permissões */}
      {showOverlayPermission && (
        <div className="fixed inset-0 z-50 bg-black bg-opacity-50 flex items-center justify-center p-4">
          <div className="bg-white rounded-3xl p-6 max-w-sm w-full shadow-2xl">
            <div className="text-center">
              <div 
                className="w-16 h-16 mx-auto mb-4 rounded-full flex items-center justify-center"
                style={{ backgroundColor: '#FF6B35' }}
              >
                <Smartphone size={32} className="text-white" />
              </div>
              
              <h3 className="text-xl font-bold text-gray-900 mb-2">
                Permissões do App
              </h3>
              
              <p className="text-gray-600 mb-6 text-sm">
                Para melhor experiência, ative as permissões de notificação e sobreposição de tela.
              </p>
              
              <div className="space-y-3">
                <button
                  onClick={requestOverlayPermission}
                  className="w-full py-3 px-4 text-white rounded-xl font-semibold transition-all duration-200 shadow-lg hover:opacity-90"
                  style={{ backgroundColor: '#FF6B35' }}
                >
                  Ativar Permissões
                </button>
                
                <button
                  onClick={() => setShowOverlayPermission(false)}
                  className="w-full py-3 px-4 bg-gray-100 hover:bg-gray-200 text-gray-700 rounded-xl font-semibold transition-all duration-200"
                >
                  Agora Não
                </button>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Notificação */}
      {notification && (
        <div className="fixed top-24 left-4 right-4 z-50 max-w-sm mx-auto">
          <div 
            className="bg-white rounded-2xl p-4 shadow-xl border-l-4 transform animate-slide-down"
            style={{
              borderLeftColor: notification.type === 'success' ? '#10b981' : 
                              notification.type === 'error' ? '#ef4444' : 
                              notification.type === 'warning' ? '#f59e0b' : '#3b82f6'
            }}
          >
            <div className="flex items-center gap-3">
              <div 
                className="w-10 h-10 rounded-full flex items-center justify-center text-white"
                style={{
                  backgroundColor: notification.type === 'success' ? '#10b981' : 
                                  notification.type === 'error' ? '#ef4444' : 
                                  notification.type === 'warning' ? '#f59e0b' : '#3b82f6'
                }}
              >
                {notification.type === 'success' ? <CheckCircle size={20} /> : 
                 notification.type === 'error' ? <XCircle size={20} /> : 
                 <Search size={20} />}
              </div>
              <div className="flex-1">
                <div className="font-semibold text-gray-900">{notification.title}</div>
                <div className="text-sm text-gray-600 mt-1">{notification.message}</div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default MainScreen;