import React, { useState, useEffect } from 'react';
import { ArrowLeft, TrendingUp, Calendar, DollarSign, Clock } from 'lucide-react';
import { driverService } from '../../services/api';

const EarningsScreen = ({ user, onBack }) => {
  const [earnings, setEarnings] = useState({
    saldo: 0,
    ganho_hoje: 0,
    corridas_hoje: 0,
    ganho_semana: 0,
    corridas_semana: 0,
    ganho_mes: 0,
    corridas_mes: 0
  });
  const [loading, setLoading] = useState(true);
  const [recentRides, setRecentRides] = useState([]);

  useEffect(() => {
    loadEarningsData();
  }, []);

  const loadEarningsData = async () => {
    try {
      setLoading(true);
      
      if (user?.id) {
        // Carregar saldo
        const balanceResponse = await driverService.getBalance(user.id);
        if (balanceResponse.data.success) {
          setEarnings(prev => ({
            ...prev,
            saldo: parseFloat(balanceResponse.data.saldo) || 0
          }));
        }
        
        // Carregar estatísticas do dia
        const statsResponse = await driverService.getDayStats(user.id);
        if (statsResponse.data.success) {
          setEarnings(prev => ({
            ...prev,
            ganho_hoje: parseFloat(statsResponse.data.ganho_hoje) || 0,
            corridas_hoje: parseInt(statsResponse.data.corridas_hoje) || 0
          }));
        }
        
        // TODO: Implementar estatísticas de semana e mês no backend
        // Por enquanto, usar dados simulados
        setEarnings(prev => ({
          ...prev,
          ganho_semana: prev.ganho_hoje * 5.2, // Simulação
          corridas_semana: prev.corridas_hoje * 5,
          ganho_mes: prev.ganho_hoje * 22, // Simulação
          corridas_mes: prev.corridas_hoje * 20
        }));
      }
    } catch (error) {
      console.error('Erro ao carregar dados de ganhos:', error);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <div className="w-12 h-12 border-4 border-orange-500 border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
          <p className="text-gray-600">Carregando ganhos...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Header */}
      <div className="bg-gradient-to-r from-orange-500 to-orange-600 shadow-lg">
        <div className="flex items-center justify-between p-4">
          <button
            onClick={onBack}
            className="w-12 h-12 bg-white bg-opacity-20 rounded-xl flex items-center justify-center hover:bg-opacity-30 transition-all"
          >
            <ArrowLeft size={20} className="text-white" />
          </button>
          <div className="text-center">
            <h1 className="text-xl font-bold text-white">💰 Meus Ganhos</h1>
            <p className="text-orange-100 text-sm">Acompanhe seus rendimentos</p>
          </div>
          <div className="w-12"></div>
        </div>
      </div>

      <div className="p-4 space-y-6">
        {/* Saldo Principal - Centralizado */}
        <div className="bg-gradient-to-r from-orange-500 to-orange-600 rounded-3xl p-8 text-white text-center shadow-xl">
          <div className="mb-6">
            <p className="text-orange-100 text-sm mb-2">SALDO TOTAL DISPONÍVEL</p>
            <p className="text-4xl font-bold mb-2">R$ {earnings.saldo.toFixed(2)}</p>
            <p className="text-orange-100 text-sm">Pronto para saque</p>
          </div>
          <button className="w-full bg-white bg-opacity-20 hover:bg-opacity-30 text-white font-semibold py-4 rounded-2xl transition-all text-lg">
            💳 Sacar Dinheiro
          </button>
        </div>

        {/* Ganhos por Período */}
        <div className="bg-white rounded-2xl p-6 shadow-lg">
          <h3 className="text-xl font-bold text-gray-900 mb-6 text-center">📊 Ganhos por Período</h3>
          
          <div className="space-y-4">
            {/* Ganhos de Hoje */}
            <div className="bg-gradient-to-r from-green-50 to-green-100 rounded-xl p-5">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-12 h-12 bg-green-500 rounded-full flex items-center justify-center">
                    <Calendar size={20} className="text-white" />
                  </div>
                  <div>
                    <p className="font-bold text-lg text-gray-900">Ganhos de Hoje</p>
                    <p className="text-sm text-gray-600">{earnings.corridas_hoje} corridas realizadas</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-bold text-2xl text-green-600">R$ {earnings.ganho_hoje.toFixed(2)}</p>
                  <p className="text-sm text-gray-500">
                    Média: R$ {earnings.corridas_hoje > 0 ? (earnings.ganho_hoje / earnings.corridas_hoje).toFixed(2) : '0.00'}
                  </p>
                </div>
              </div>
            </div>

            {/* Ganhos da Semana */}
            <div className="bg-gradient-to-r from-blue-50 to-blue-100 rounded-xl p-5">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-12 h-12 bg-blue-500 rounded-full flex items-center justify-center">
                    <TrendingUp size={20} className="text-white" />
                  </div>
                  <div>
                    <p className="font-bold text-lg text-gray-900">Ganhos da Semana</p>
                    <p className="text-sm text-gray-600">{earnings.corridas_semana} corridas realizadas</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-bold text-2xl text-blue-600">R$ {earnings.ganho_semana.toFixed(2)}</p>
                  <p className="text-sm text-gray-500">
                    Média: R$ {earnings.corridas_semana > 0 ? (earnings.ganho_semana / earnings.corridas_semana).toFixed(2) : '0.00'}
                  </p>
                </div>
              </div>
            </div>

            {/* Ganhos do Mês */}
            <div className="bg-gradient-to-r from-purple-50 to-purple-100 rounded-xl p-5">
              <div className="flex items-center justify-between">
                <div className="flex items-center gap-3">
                  <div className="w-12 h-12 bg-purple-500 rounded-full flex items-center justify-center">
                    <Clock size={20} className="text-white" />
                  </div>
                  <div>
                    <p className="font-bold text-lg text-gray-900">Ganhos do Mês</p>
                    <p className="text-sm text-gray-600">{earnings.corridas_mes} corridas realizadas</p>
                  </div>
                </div>
                <div className="text-right">
                  <p className="font-bold text-2xl text-purple-600">R$ {earnings.ganho_mes.toFixed(2)}</p>
                  <p className="text-sm text-gray-500">
                    Média: R$ {earnings.corridas_mes > 0 ? (earnings.ganho_mes / earnings.corridas_mes).toFixed(2) : '0.00'}
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Análise Financeira */}
        <div className="bg-white rounded-2xl p-6 shadow-lg">
          <h3 className="text-xl font-bold text-gray-900 mb-6 text-center">💰 Análise Financeira</h3>
          
          <div className="grid grid-cols-2 gap-4 mb-6">
            <div className="bg-gray-50 rounded-xl p-4 text-center">
              <p className="text-sm text-gray-600 mb-1">Valor Bruto Hoje</p>
              <p className="font-bold text-lg text-gray-900">R$ {(earnings.ganho_hoje / 0.85).toFixed(2)}</p>
            </div>
            <div className="bg-gray-50 rounded-xl p-4 text-center">
              <p className="text-sm text-gray-600 mb-1">Taxa Plataforma</p>
              <p className="font-bold text-lg text-orange-600">- R$ {(earnings.ganho_hoje * 0.15 / 0.85).toFixed(2)}</p>
            </div>
          </div>

          <div className="space-y-4">
            <div className="flex items-center justify-between py-3 border-b border-gray-100">
              <div>
                <p className="font-medium text-gray-900">💵 Ganho Líquido Hoje</p>
                <p className="text-sm text-gray-500">{earnings.corridas_hoje} corridas • Após taxas</p>
              </div>
              <p className="font-bold text-green-600 text-lg">R$ {earnings.ganho_hoje.toFixed(2)}</p>
            </div>

            <div className="flex items-center justify-between py-3 border-b border-gray-100">
              <div>
                <p className="font-medium text-gray-900">📊 Média por Corrida</p>
                <p className="text-sm text-gray-500">Valor líquido médio</p>
              </div>
              <p className="font-bold text-blue-600 text-lg">
                R$ {earnings.corridas_hoje > 0 ? (earnings.ganho_hoje / earnings.corridas_hoje).toFixed(2) : '0.00'}
              </p>
            </div>

            <div className="flex items-center justify-between py-3">
              <div>
                <p className="font-medium text-gray-900">⏱️ Tempo Online Estimado</p>
                <p className="text-sm text-gray-500">Baseado nas corridas</p>
              </div>
              <p className="font-bold text-purple-600 text-lg">
                {Math.round(earnings.corridas_hoje * 0.75)}h {Math.round((earnings.corridas_hoje * 0.75 % 1) * 60)}min
              </p>
            </div>
          </div>
        </div>

        {/* Metas Diárias */}
        <div className="bg-white rounded-2xl p-6 shadow-lg">
          <h3 className="text-xl font-bold text-gray-900 mb-6 text-center">🎯 Metas do Dia</h3>
          
          <div className="space-y-6">
            {/* Meta de Corridas */}
            <div>
              <div className="flex items-center justify-between mb-3">
                <div className="flex items-center gap-2">
                  <span className="text-lg">🚗</span>
                  <p className="font-medium text-gray-700">Corridas Realizadas</p>
                </div>
                <p className="font-bold text-gray-900">{earnings.corridas_hoje}/10</p>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-3">
                <div 
                  className="bg-gradient-to-r from-orange-400 to-orange-500 h-3 rounded-full transition-all duration-500 flex items-center justify-end pr-2"
                  style={{ width: `${Math.min((earnings.corridas_hoje / 10) * 100, 100)}%` }}
                >
                  {earnings.corridas_hoje >= 5 && (
                    <span className="text-white text-xs font-bold">
                      {Math.round((earnings.corridas_hoje / 10) * 100)}%
                    </span>
                  )}
                </div>
              </div>
              <p className="text-sm text-gray-500 mt-1">
                {earnings.corridas_hoje >= 10 ? '🎉 Meta atingida!' : `Faltam ${10 - earnings.corridas_hoje} corridas`}
              </p>
            </div>

            {/* Meta de Ganhos */}
            <div>
              <div className="flex items-center justify-between mb-3">
                <div className="flex items-center gap-2">
                  <span className="text-lg">💰</span>
                  <p className="font-medium text-gray-700">Ganhos do Dia</p>
                </div>
                <p className="font-bold text-gray-900">R$ {earnings.ganho_hoje.toFixed(2)}/200</p>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-3">
                <div 
                  className="bg-gradient-to-r from-green-400 to-green-500 h-3 rounded-full transition-all duration-500 flex items-center justify-end pr-2"
                  style={{ width: `${Math.min((earnings.ganho_hoje / 200) * 100, 100)}%` }}
                >
                  {earnings.ganho_hoje >= 50 && (
                    <span className="text-white text-xs font-bold">
                      {Math.round((earnings.ganho_hoje / 200) * 100)}%
                    </span>
                  )}
                </div>
              </div>
              <p className="text-sm text-gray-500 mt-1">
                {earnings.ganho_hoje >= 200 ? '🎉 Meta atingida!' : `Faltam R$ ${(200 - earnings.ganho_hoje).toFixed(2)}`}
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default EarningsScreen;