const CACHE_NAME = 'rideshare-motorista-v1.0.0';
const STATIC_CACHE = 'rideshare-static-v1';
const DYNAMIC_CACHE = 'rideshare-dynamic-v1';

// Arquivos para cache estático
const STATIC_FILES = [
  '/',
  '/index.html',
  '/manifest.json',
  '/icons/icon-192x192.png',
  '/icons/icon-512x512.png'
];

// URLs da API para cache dinâmico
const API_URLS = [
  '/plataforma/api/auth.php',
  '/plataforma/api/corridas.php',
  '/plataforma/api/motoristas.php'
];

// Instalar Service Worker
self.addEventListener('install', (event) => {
  console.log('🔧 Service Worker: Instalando...');
  
  event.waitUntil(
    caches.open(STATIC_CACHE)
      .then((cache) => {
        console.log('📦 Service Worker: Cache estático criado');
        return cache.addAll(STATIC_FILES);
      })
      .then(() => {
        console.log('✅ Service Worker: Instalado com sucesso');
        return self.skipWaiting();
      })
      .catch((error) => {
        console.error('❌ Service Worker: Erro na instalação:', error);
      })
  );
});

// Ativar Service Worker
self.addEventListener('activate', (event) => {
  console.log('🚀 Service Worker: Ativando...');
  
  event.waitUntil(
    caches.keys()
      .then((cacheNames) => {
        return Promise.all(
          cacheNames.map((cacheName) => {
            if (cacheName !== STATIC_CACHE && cacheName !== DYNAMIC_CACHE) {
              console.log('🗑️ Service Worker: Removendo cache antigo:', cacheName);
              return caches.delete(cacheName);
            }
          })
        );
      })
      .then(() => {
        console.log('✅ Service Worker: Ativado com sucesso');
        return self.clients.claim();
      })
  );
});

// Interceptar requisições
self.addEventListener('fetch', (event) => {
  const { request } = event;
  const url = new URL(request.url);
  
  // Estratégia para arquivos estáticos
  if (STATIC_FILES.some(file => url.pathname.endsWith(file))) {
    event.respondWith(cacheFirst(request));
    return;
  }
  
  // Estratégia para APIs
  if (API_URLS.some(apiUrl => url.pathname.includes(apiUrl))) {
    event.respondWith(networkFirst(request));
    return;
  }
  
  // Estratégia padrão
  event.respondWith(networkFirst(request));
});

// Estratégia Cache First (para arquivos estáticos)
async function cacheFirst(request) {
  try {
    const cachedResponse = await caches.match(request);
    if (cachedResponse) {
      return cachedResponse;
    }
    
    const networkResponse = await fetch(request);
    const cache = await caches.open(STATIC_CACHE);
    cache.put(request, networkResponse.clone());
    
    return networkResponse;
  } catch (error) {
    console.error('Cache First falhou:', error);
    return new Response('Offline', { status: 503 });
  }
}

// Estratégia Network First (para APIs)
async function networkFirst(request) {
  try {
    const networkResponse = await fetch(request);
    
    // Cache apenas respostas bem-sucedidas
    if (networkResponse.ok) {
      const cache = await caches.open(DYNAMIC_CACHE);
      cache.put(request, networkResponse.clone());
    }
    
    return networkResponse;
  } catch (error) {
    console.log('Network falhou, tentando cache:', error);
    
    const cachedResponse = await caches.match(request);
    if (cachedResponse) {
      return cachedResponse;
    }
    
    // Resposta offline para APIs
    if (request.url.includes('/api/')) {
      return new Response(JSON.stringify({
        success: false,
        message: 'Sem conexão com a internet',
        offline: true
      }), {
        status: 503,
        headers: { 'Content-Type': 'application/json' }
      });
    }
    
    return new Response('Sem conexão', { status: 503 });
  }
}

// Notificações Push
self.addEventListener('push', (event) => {
  console.log('📱 Push recebido:', event);
  
  const options = {
    body: event.data ? event.data.text() : 'Nova corrida disponível!',
    icon: '/icons/icon-192x192.png',
    badge: '/icons/icon-72x72.png',
    vibrate: [200, 100, 200],
    data: {
      url: '/'
    },
    actions: [
      {
        action: 'accept',
        title: 'Aceitar',
        icon: '/icons/icon-96x96.png'
      },
      {
        action: 'reject',
        title: 'Recusar',
        icon: '/icons/icon-96x96.png'
      }
    ]
  };
  
  event.waitUntil(
    self.registration.showNotification('RideShare - Nova Corrida!', options)
  );
});

// Clique em notificação
self.addEventListener('notificationclick', (event) => {
  console.log('🔔 Notificação clicada:', event);
  
  event.notification.close();
  
  if (event.action === 'accept') {
    // Lógica para aceitar corrida
    event.waitUntil(
      clients.openWindow('/?action=accept')
    );
  } else if (event.action === 'reject') {
    // Lógica para recusar corrida
    console.log('Corrida recusada via notificação');
  } else {
    // Abrir app
    event.waitUntil(
      clients.openWindow('/')
    );
  }
});

// Sincronização em background
self.addEventListener('sync', (event) => {
  console.log('🔄 Background Sync:', event.tag);
  
  if (event.tag === 'location-sync') {
    event.waitUntil(syncLocation());
  }
});

// Função para sincronizar localização
async function syncLocation() {
  try {
    // Implementar sincronização de localização quando voltar online
    console.log('📍 Sincronizando localização...');
  } catch (error) {
    console.error('Erro na sincronização:', error);
  }
}

// Mensagens do cliente
self.addEventListener('message', (event) => {
  console.log('💬 Mensagem recebida:', event.data);
  
  if (event.data && event.data.type === 'SKIP_WAITING') {
    self.skipWaiting();
  }
});

console.log('🚗 RideShare Motorista Service Worker carregado!');