<?php
require_once __DIR__ . '/../config/database.php';

// Função para obter configurações do banco de dados
function getConfig($chave, $default = null) {
    static $cache = [];
    
    if (isset($cache[$chave])) {
        return $cache[$chave];
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("SELECT valor, tipo FROM configuracoes WHERE chave = ?");
        $stmt->execute([$chave]);
        $config = $stmt->fetch();
        
        if ($config) {
            $valor = $config['valor'];
            
            // Converter tipo
            switch ($config['tipo']) {
                case 'numero':
                    $valor = is_numeric($valor) ? (float)$valor : $valor;
                    break;
                case 'boolean':
                    $valor = ($valor === 'true' || $valor === '1');
                    break;
                case 'json':
                    $valor = json_decode($valor, true);
                    break;
            }
            
            $cache[$chave] = $valor;
            return $valor;
        }
        
        return $default;
    } catch (Exception $e) {
        return $default;
    }
}

// Função para atualizar configurações
function updateConfig($chave, $valor) {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("UPDATE configuracoes SET valor = ? WHERE chave = ?");
        return $stmt->execute([$valor, $chave]);
    } catch (Exception $e) {
        return false;
    }
}

// Função para gerar hash de senha
function hashPassword($senha) {
    return password_hash($senha, PASSWORD_BCRYPT);
}

// Função para verificar senha
function verifyPassword($senha, $hash) {
    return password_verify($senha, $hash);
}

// Função para gerar token único
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

// Função para calcular distância entre dois pontos (Haversine)
function calcularDistancia($lat1, $lon1, $lat2, $lon2) {
    $raioTerra = 6371; // km
    
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    
    $a = sin($dLat/2) * sin($dLat/2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLon/2) * sin($dLon/2);
    
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    $distancia = $raioTerra * $c;
    
    return round($distancia, 2);
}

// Função para calcular tarifa
function calcularTarifa($categoria_id, $distancia_km, $duracao_minutos, $multiplicador = 1.0) {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("SELECT tarifa_base, tarifa_por_km, tarifa_por_minuto FROM categorias_veiculo WHERE id = ?");
        $stmt->execute([$categoria_id]);
        $categoria = $stmt->fetch();
        
        if (!$categoria) {
            return false;
        }
        
        $tarifa_base = $categoria['tarifa_base'];
        $tarifa_distancia = $distancia_km * $categoria['tarifa_por_km'];
        $tarifa_tempo = $duracao_minutos * $categoria['tarifa_por_minuto'];
        
        $subtotal = ($tarifa_base + $tarifa_distancia + $tarifa_tempo) * $multiplicador;
        $taxa_plataforma = $subtotal * (getConfig('platform_fee_percent', 20) / 100);
        $valor_motorista = $subtotal - $taxa_plataforma;
        
        return [
            'tarifa_base' => round($tarifa_base, 2),
            'tarifa_distancia' => round($tarifa_distancia, 2),
            'tarifa_tempo' => round($tarifa_tempo, 2),
            'multiplicador' => $multiplicador,
            'valor_total' => round($subtotal, 2),
            'taxa_plataforma' => round($taxa_plataforma, 2),
            'valor_motorista' => round($valor_motorista, 2)
        ];
    } catch (Exception $e) {
        return false;
    }
}

// Função para obter multiplicador de tarifa dinâmica
function obterMultiplicadorDinamico($categoria_id, $latitude, $longitude) {
    if (!getConfig('dynamic_pricing_enabled', true)) {
        return 1.0;
    }
    
    try {
        $db = new Database();
        $conn = $db->connect();
        
        // Buscar tarifas dinâmicas ativas na região
        $stmt = $conn->prepare("
            SELECT multiplicador, 
                   (6371 * acos(cos(radians(?)) * cos(radians(latitude)) * 
                   cos(radians(longitude) - radians(?)) + sin(radians(?)) * 
                   sin(radians(latitude)))) AS distancia
            FROM tarifa_dinamica
            WHERE categoria_veiculo_id = ?
            AND ativo = TRUE
            AND expira_em > NOW()
            HAVING distancia <= raio_km
            ORDER BY multiplicador DESC
            LIMIT 1
        ");
        
        $stmt->execute([$latitude, $longitude, $latitude, $categoria_id]);
        $tarifa = $stmt->fetch();
        
        if ($tarifa) {
            $max = getConfig('surge_multiplier_max', 3.0);
            return min($tarifa['multiplicador'], $max);
        }
        
        return 1.0;
    } catch (Exception $e) {
        return 1.0;
    }
}

// Função para registrar log
function registrarLog($tipo_usuario, $usuario_id, $acao, $descricao = '') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
        
        $stmt = $conn->prepare("
            INSERT INTO logs_sistema (tipo_usuario, usuario_id, acao, descricao, ip_address, user_agent)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([$tipo_usuario, $usuario_id, $acao, $descricao, $ip, $user_agent]);
    } catch (Exception $e) {
        return false;
    }
}

// Função para enviar notificação
function enviarNotificacao($tipo_usuario, $usuario_id, $titulo, $mensagem, $tipo = 'info') {
    try {
        $db = new Database();
        $conn = $db->connect();
        
        $stmt = $conn->prepare("
            INSERT INTO notificacoes (tipo_usuario, usuario_id, titulo, mensagem, tipo)
            VALUES (?, ?, ?, ?, ?)
        ");
        
        return $stmt->execute([$tipo_usuario, $usuario_id, $titulo, $mensagem, $tipo]);
    } catch (Exception $e) {
        return false;
    }
}

// Função para resposta JSON
function jsonResponse($data, $status = 200) {
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

// Função para validar email
function validarEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Função para validar CPF
function validarCPF($cpf) {
    $cpf = preg_replace('/[^0-9]/', '', $cpf);
    
    if (strlen($cpf) != 11 || preg_match('/(\d)\1{10}/', $cpf)) {
        return false;
    }
    
    for ($t = 9; $t < 11; $t++) {
        $d = 0;
        for ($c = 0; $c < $t; $c++) {
            $d += $cpf[$c] * (($t + 1) - $c);
        }
        $d = ((10 * $d) % 11) % 10;
        if ($cpf[$c] != $d) {
            return false;
        }
    }
    return true;
}

// Função para formatar telefone
function formatarTelefone($telefone) {
    $telefone = preg_replace('/[^0-9]/', '', $telefone);
    
    if (strlen($telefone) == 11) {
        return preg_replace('/(\d{2})(\d{5})(\d{4})/', '($1) $2-$3', $telefone);
    } elseif (strlen($telefone) == 10) {
        return preg_replace('/(\d{2})(\d{4})(\d{4})/', '($1) $2-$3', $telefone);
    }
    
    return $telefone;
}
