-- Criação do Banco de Dados
CREATE DATABASE IF NOT EXISTS rideshare_platform CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE rideshare_platform;

-- Tabela de Usuários (Passageiros)
CREATE TABLE usuarios (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    telefone VARCHAR(20) NOT NULL,
    senha VARCHAR(255) NOT NULL,
    foto_perfil VARCHAR(255),
    cpf VARCHAR(14) UNIQUE,
    data_nascimento DATE,
    status ENUM('ativo', 'bloqueado', 'inativo') DEFAULT 'ativo',
    avaliacao_media DECIMAL(3,2) DEFAULT 5.00,
    total_avaliacoes INT DEFAULT 0,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_status (status)
) ENGINE=InnoDB;

-- Tabela de Categorias de Veículos
CREATE TABLE categorias_veiculo (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(50) NOT NULL,
    descricao TEXT,
    tipo ENUM('carro', 'moto', 'van', 'caminhao') NOT NULL,
    capacidade_passageiros INT,
    capacidade_carga_kg DECIMAL(10,2),
    tarifa_base DECIMAL(10,2) NOT NULL,
    tarifa_por_km DECIMAL(10,2) NOT NULL,
    tarifa_por_minuto DECIMAL(10,2) NOT NULL,
    icone VARCHAR(255),
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB;

-- Tabela de Motoristas
CREATE TABLE motoristas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    telefone VARCHAR(20) NOT NULL,
    senha VARCHAR(255) NOT NULL,
    foto_perfil VARCHAR(255),
    cpf VARCHAR(14) UNIQUE NOT NULL,
    cnh VARCHAR(20) UNIQUE NOT NULL,
    data_nascimento DATE NOT NULL,
    categoria_veiculo_id INT NOT NULL,
    placa_veiculo VARCHAR(10) NOT NULL,
    modelo_veiculo VARCHAR(100) NOT NULL,
    cor_veiculo VARCHAR(50) NOT NULL,
    ano_veiculo INT NOT NULL,
    status ENUM('ativo', 'bloqueado', 'inativo', 'pendente') DEFAULT 'pendente',
    disponivel BOOLEAN DEFAULT FALSE,
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    ultima_localizacao TIMESTAMP,
    avaliacao_media DECIMAL(3,2) DEFAULT 5.00,
    total_avaliacoes INT DEFAULT 0,
    total_corridas INT DEFAULT 0,
    saldo DECIMAL(10,2) DEFAULT 0.00,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (categoria_veiculo_id) REFERENCES categorias_veiculo(id),
    INDEX idx_email (email),
    INDEX idx_status (status),
    INDEX idx_disponivel (disponivel),
    INDEX idx_localizacao (latitude, longitude)
) ENGINE=InnoDB;

-- Tabela de Administradores
CREATE TABLE administradores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    nivel ENUM('super', 'admin', 'moderador') DEFAULT 'moderador',
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_email (email)
) ENGINE=InnoDB;

-- Tabela de Corridas/Entregas
CREATE TABLE corridas (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    motorista_id INT,
    categoria_veiculo_id INT NOT NULL,
    tipo ENUM('passageiro', 'entrega') DEFAULT 'passageiro',
    
    -- Localização
    origem_latitude DECIMAL(10,8) NOT NULL,
    origem_longitude DECIMAL(11,8) NOT NULL,
    origem_endereco TEXT NOT NULL,
    destino_latitude DECIMAL(10,8) NOT NULL,
    destino_longitude DECIMAL(11,8) NOT NULL,
    destino_endereco TEXT NOT NULL,
    
    -- Informações da Entrega (se tipo = entrega)
    descricao_entrega TEXT,
    nome_destinatario VARCHAR(100),
    telefone_destinatario VARCHAR(20),
    
    -- Valores
    distancia_km DECIMAL(10,2),
    duracao_minutos INT,
    tarifa_base DECIMAL(10,2),
    tarifa_distancia DECIMAL(10,2),
    tarifa_tempo DECIMAL(10,2),
    multiplicador_dinamico DECIMAL(3,2) DEFAULT 1.00,
    valor_total DECIMAL(10,2),
    taxa_plataforma DECIMAL(10,2),
    valor_motorista DECIMAL(10,2),
    
    -- Pagamento
    metodo_pagamento ENUM('dinheiro', 'cartao', 'pix'),
    status_pagamento ENUM('pendente', 'pago', 'cancelado') DEFAULT 'pendente',
    
    -- Status
    status ENUM('solicitada', 'aceita', 'em_andamento', 'concluida', 'cancelada') DEFAULT 'solicitada',
    cancelado_por ENUM('usuario', 'motorista', 'sistema'),
    motivo_cancelamento TEXT,
    
    -- Avaliações
    avaliacao_usuario INT,
    comentario_usuario TEXT,
    avaliacao_motorista INT,
    comentario_motorista TEXT,
    
    -- Timestamps
    solicitada_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    aceita_em TIMESTAMP NULL,
    iniciada_em TIMESTAMP NULL,
    concluida_em TIMESTAMP NULL,
    cancelada_em TIMESTAMP NULL,
    
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id),
    FOREIGN KEY (motorista_id) REFERENCES motoristas(id),
    FOREIGN KEY (categoria_veiculo_id) REFERENCES categorias_veiculo(id),
    INDEX idx_usuario (usuario_id),
    INDEX idx_motorista (motorista_id),
    INDEX idx_status (status),
    INDEX idx_data (solicitada_em)
) ENGINE=InnoDB;

-- Tabela de Tarifa Dinâmica
CREATE TABLE tarifa_dinamica (
    id INT AUTO_INCREMENT PRIMARY KEY,
    categoria_veiculo_id INT NOT NULL,
    latitude DECIMAL(10,8) NOT NULL,
    longitude DECIMAL(11,8) NOT NULL,
    raio_km DECIMAL(5,2) NOT NULL,
    multiplicador DECIMAL(3,2) NOT NULL,
    demanda_atual INT DEFAULT 0,
    motoristas_disponiveis INT DEFAULT 0,
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    expira_em TIMESTAMP NOT NULL,
    FOREIGN KEY (categoria_veiculo_id) REFERENCES categorias_veiculo(id),
    INDEX idx_localizacao (latitude, longitude),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB;

-- Tabela de Métodos de Pagamento dos Usuários
CREATE TABLE metodos_pagamento (
    id INT AUTO_INCREMENT PRIMARY KEY,
    usuario_id INT NOT NULL,
    tipo ENUM('cartao', 'pix') NOT NULL,
    
    -- Dados do Cartão (criptografados)
    numero_cartao_hash VARCHAR(255),
    nome_titular VARCHAR(100),
    validade VARCHAR(7),
    
    -- Dados PIX
    chave_pix VARCHAR(255),
    tipo_chave_pix ENUM('cpf', 'email', 'telefone', 'aleatoria'),
    
    padrao BOOLEAN DEFAULT FALSE,
    ativo BOOLEAN DEFAULT TRUE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id),
    INDEX idx_usuario (usuario_id)
) ENGINE=InnoDB;

-- Tabela de Transações Financeiras
CREATE TABLE transacoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    corrida_id INT,
    motorista_id INT,
    tipo ENUM('corrida', 'saque', 'taxa', 'estorno') NOT NULL,
    valor DECIMAL(10,2) NOT NULL,
    descricao TEXT,
    status ENUM('pendente', 'concluida', 'cancelada') DEFAULT 'pendente',
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (corrida_id) REFERENCES corridas(id),
    FOREIGN KEY (motorista_id) REFERENCES motoristas(id),
    INDEX idx_motorista (motorista_id),
    INDEX idx_tipo (tipo)
) ENGINE=InnoDB;

-- Tabela de Notificações
CREATE TABLE notificacoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tipo_usuario ENUM('usuario', 'motorista', 'admin') NOT NULL,
    usuario_id INT,
    titulo VARCHAR(255) NOT NULL,
    mensagem TEXT NOT NULL,
    tipo ENUM('info', 'alerta', 'corrida', 'pagamento') DEFAULT 'info',
    lida BOOLEAN DEFAULT FALSE,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_usuario (tipo_usuario, usuario_id),
    INDEX idx_lida (lida)
) ENGINE=InnoDB;

-- Tabela de Logs do Sistema
CREATE TABLE logs_sistema (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tipo_usuario ENUM('usuario', 'motorista', 'admin') NOT NULL,
    usuario_id INT,
    acao VARCHAR(100) NOT NULL,
    descricao TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    criado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_usuario (tipo_usuario, usuario_id),
    INDEX idx_data (criado_em)
) ENGINE=InnoDB;

-- Inserir Categorias Padrão
INSERT INTO categorias_veiculo (nome, descricao, tipo, capacidade_passageiros, tarifa_base, tarifa_por_km, tarifa_por_minuto, ativo) VALUES
('Moto', 'Viagem rápida e econômica', 'moto', 1, 3.00, 1.50, 0.30, TRUE),
('Carro Popular', 'Conforto e economia', 'carro', 4, 5.00, 2.50, 0.50, TRUE),
('Carro Confort', 'Mais espaço e conforto', 'carro', 4, 8.00, 3.50, 0.70, TRUE),
('Van', 'Para grupos e bagagens', 'van', 6, 12.00, 4.00, 1.00, TRUE),
('Moto Entrega', 'Entregas rápidas até 10kg', 'moto', 0, 4.00, 2.00, 0.40, TRUE),
('Carro Entrega', 'Entregas até 50kg', 'carro', 0, 7.00, 3.00, 0.60, TRUE),
('Van Entrega', 'Entregas grandes até 500kg', 'van', 0, 15.00, 5.00, 1.20, TRUE);

-- Tabela de Configurações do Sistema
CREATE TABLE configuracoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    chave VARCHAR(100) UNIQUE NOT NULL,
    valor TEXT,
    descricao TEXT,
    tipo ENUM('texto', 'numero', 'boolean', 'json') DEFAULT 'texto',
    atualizado_em TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_chave (chave)
) ENGINE=InnoDB;

-- Inserir Configurações Padrão
INSERT INTO configuracoes (chave, valor, descricao, tipo) VALUES
('google_maps_api_key', 'AIzaSyBQyClNByvKjHubTBizQ1GquLjhwSyHz2k', 'Chave da API do Google Maps', 'texto'),
('base_fare', '5.00', 'Tarifa base padrão', 'numero'),
('price_per_km', '2.50', 'Preço por quilômetro', 'numero'),
('price_per_minute', '0.50', 'Preço por minuto', 'numero'),
('platform_fee_percent', '20', 'Taxa da plataforma em %', 'numero'),
('cancellation_fee', '3.00', 'Taxa de cancelamento', 'numero'),
('dynamic_pricing_enabled', 'true', 'Tarifa dinâmica ativada', 'boolean'),
('surge_multiplier_max', '3.0', 'Multiplicador máximo de tarifa', 'numero');

-- Inserir Admin Padrão (senha: 123456)
INSERT INTO administradores (nome, email, senha, nivel) VALUES
('Administrador', 'admin@rideshare.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'super');
